<?php
session_start();
require_once 'includes/auth_check.php';
require_once '../config/database.php';

$page = 'deposit';
$database = new Database();
$db = $database->getConnection();

// Auto-create tables if they don't exist
$tableCreationQueries = [
    'crypto_wallets' => "CREATE TABLE crypto_wallets (id INT AUTO_INCREMENT PRIMARY KEY, currency VARCHAR(20) NOT NULL, wallet_address VARCHAR(255) NOT NULL, network VARCHAR(50) DEFAULT NULL, is_active BOOLEAN DEFAULT TRUE, created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP)",
    'settings' => "CREATE TABLE settings (id INT AUTO_INCREMENT PRIMARY KEY, setting_key VARCHAR(100) NOT NULL UNIQUE, setting_value TEXT, created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP)",
    'crypto_deposits' => "CREATE TABLE crypto_deposits (id INT AUTO_INCREMENT PRIMARY KEY, user_id INT NOT NULL, currency VARCHAR(20) NOT NULL, amount DECIMAL(18, 8) NOT NULL, wallet_address VARCHAR(255) NOT NULL, transaction_hash VARCHAR(255) NOT NULL, network VARCHAR(50) DEFAULT NULL, screenshot VARCHAR(255) DEFAULT NULL, status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending', admin_notes TEXT DEFAULT NULL, admin_id INT DEFAULT NULL, created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP, FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE)",
    'admin_notifications' => "CREATE TABLE admin_notifications (id INT AUTO_INCREMENT PRIMARY KEY, admin_id INT DEFAULT NULL, type VARCHAR(50) NOT NULL, title VARCHAR(255) NOT NULL, message TEXT NOT NULL, link VARCHAR(255) DEFAULT '#', is_read BOOLEAN DEFAULT FALSE, created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP)"
];

foreach ($tableCreationQueries as $tableName => $createQuery) {
    try {
        $stmt = $db->prepare("SELECT 1 FROM $tableName LIMIT 1");
        $stmt->execute();
    } catch (Exception $e) {
        // Table does not exist, create it
        try {
            $db->exec($createQuery);
            // Insert default settings if it's the settings table
            if ($tableName === 'settings') {
                $defaultSettings = [
                    ['btc_deposit_instructions', 'Send Bitcoin to the address below. After sending, enter the transaction hash/ID as proof of payment.'],
                    ['usdt_deposit_instructions', 'Send USDT to the address below. Make sure to select the correct network. After sending, enter the transaction hash/ID as proof of payment.']
                ];
                $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)");
                foreach ($defaultSettings as $setting) {
                    $stmt->execute($setting);
                }
            }
        } catch (PDOException $ex) {
            die("DB ERROR: ". $ex->getMessage());
        }
    }
}

// Get active crypto wallets
$stmt = $db->prepare("SELECT * FROM crypto_wallets WHERE is_active = 1 ORDER BY currency, network");
$stmt->execute();
$wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Group wallets by currency
$btcWallets = array_filter($wallets, fn($w) => $w['currency'] === 'BTC');
$usdtWallets = array_filter($wallets, fn($w) => $w['currency'] === 'USDT');

// Get deposit instructions
$stmt = $db->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('btc_deposit_instructions', 'usdt_deposit_instructions')");
$instructionsData = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$btcInstructions = $instructionsData['btc_deposit_instructions'] ?? 'Send Bitcoin to the address below. After sending, enter the transaction hash/ID as proof of payment.';
$usdtInstructions = $instructionsData['usdt_deposit_instructions'] ?? 'Send USDT to the address below. Make sure to select the correct network. After sending, enter the transaction hash/ID as proof of payment.';

// Handle form submission
$errors = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $currency = $_POST['currency'] ?? '';
    $amount = $_POST['amount'] ?? 0;
    $wallet_address = $_POST['wallet_address'] ?? '';
    $transaction_hash = $_POST['transaction_hash'] ?? '';
    $network = $_POST['network'] ?? null;
    
    if (empty($currency) || !in_array($currency, ['BTC', 'USDT'])) $errors[] = "Invalid currency selected.";
    if (!is_numeric($amount) || $amount <= 0) $errors[] = "Please enter a valid, positive amount.";
    if (empty($wallet_address)) $errors[] = "Wallet address is missing. Please refresh and try again.";
    if (empty($transaction_hash)) $errors[] = "Transaction hash/ID is required.";
    if ($currency === 'USDT' && empty($network)) $errors[] = "Please select a network for USDT deposit.";
    
    $screenshot = null;
    if (isset($_FILES['screenshot']) && $_FILES['screenshot']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/deposit_proofs/';
        if (!is_dir($upload_dir)) mkdir($upload_dir, 0775, true);
        $filename = uniqid() . '-' . basename($_FILES['screenshot']['name']);
        if (move_uploaded_file($_FILES['screenshot']['tmp_name'], $upload_dir . $filename)) {
            $screenshot = $filename;
        } else {
            $errors[] = "Failed to upload screenshot.";
        }
    }
    
    if (empty($errors)) {
        try {
            $stmt = $db->prepare("INSERT INTO crypto_deposits (user_id, currency, amount, wallet_address, transaction_hash, network, screenshot) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$_SESSION['user_id'], $currency, $amount, $wallet_address, $transaction_hash, $network, $screenshot]);
            
            $stmt = $db->prepare("INSERT INTO admin_notifications (type, title, message, link) VALUES ('new_deposit', 'New Crypto Deposit', ?, 'crypto_deposits.php')");
            $stmt->execute(['A new ' . htmlspecialchars($currency) . ' deposit of ' . htmlspecialchars($amount) . ' has been submitted.']);
            
            $_SESSION['success'] = "Your deposit has been submitted successfully. It will be credited to your account after confirmation.";
            header("Location: deposit.php");
            exit();
        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Get recent deposits
$stmt = $db->prepare("SELECT * FROM crypto_deposits WHERE user_id = ? ORDER BY created_at DESC LIMIT 5");
$stmt->execute([$_SESSION['user_id']]);
$recent_deposits = $stmt->fetchAll(PDO::FETCH_ASSOC);

include 'includes/header.php';
include 'includes/sidebar.php';
?>

<div class="main-content">
    <div class="container py-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h1 class="h3 mb-1">Deposit Funds</h1>
                <p class="text-muted mb-0">Add funds to your account using cryptocurrency</p>
            </div>
        </div>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class='bx bx-check-circle me-2'></i>
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class='bx bx-error-circle me-2'></i>
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo $error; ?></li>
                    <?php endforeach; ?>
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>

        <div class="dashboard-card">
            <div class="card-body">
                <?php if (empty($wallets)): ?>
                    <div class="alert alert-warning" role="alert">
                        <i class='bx bx-error-circle me-2'></i> Crypto deposits are currently unavailable. Please contact support.
                    </div>
                <?php else: ?>
                    <ul class="nav nav-tabs" id="depositTabs" role="tablist">
                        <?php if (!empty($btcWallets)): ?>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active" id="btc-tab" data-bs-toggle="tab" data-bs-target="#btc-tab-pane" type="button" role="tab" aria-controls="btc-tab-pane" aria-selected="true">
                                    <i class='bx bxl-bitcoin text-warning me-1'></i> Bitcoin (BTC)
                                </button>
                            </li>
                        <?php endif; ?>
                        
                        <?php if (!empty($usdtWallets)): ?>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link <?php echo empty($btcWallets) ? 'active' : ''; ?>" id="usdt-tab" data-bs-toggle="tab" data-bs-target="#usdt-tab-pane" type="button" role="tab" aria-controls="usdt-tab-pane" aria-selected="<?php echo empty($btcWallets) ? 'true' : 'false'; ?>">
                                    <i class='bx bx-dollar-circle text-success me-1'></i> Tether (USDT)
                                </button>
                            </li>
                        <?php endif; ?>
                    </ul>
                    
                    <div class="tab-content pt-4" id="depositTabsContent">
                        <?php if (!empty($btcWallets)): ?>
                            <div class="tab-pane fade show active" id="btc-tab-pane" role="tabpanel" aria-labelledby="btc-tab">
                                <div class="alert alert-info mb-4"><i class='bx bx-info-circle me-2'></i> <?php echo htmlspecialchars($btcInstructions); ?></div>
                                <?php foreach ($btcWallets as $wallet): ?>
                                    <div class="mb-3">
                                        <label class="form-label">BTC Wallet Address</label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" value="<?php echo htmlspecialchars($wallet['wallet_address']); ?>" readonly>
                                            <button class="btn btn-outline-secondary copy-address" type="button" data-address="<?php echo htmlspecialchars($wallet['wallet_address']); ?>"><i class='bx bx-copy'></i> Copy</button>
                                        </div>
                                    </div>
                                    <div class="text-center mb-4"><div id="btc-qrcode-<?php echo $wallet['id']; ?>" class="qrcode"></div></div>
                                <?php endforeach; ?>
                                
                                <form method="post" enctype="multipart/form-data">
                                    <input type="hidden" name="currency" value="BTC">
                                    <input type="hidden" name="wallet_address" value="<?php echo htmlspecialchars(reset($btcWallets)['wallet_address']); ?>">
                                    <div class="mb-3"><label for="btc-amount" class="form-label">Amount (BTC)</label><input type="number" class="form-control" id="btc-amount" name="amount" step="0.00000001" required></div>
                                    <div class="mb-3"><label for="btc-tx" class="form-label">Transaction Hash/ID</label><input type="text" class="form-control" id="btc-tx" name="transaction_hash" required></div>
                                    <div class="mb-3"><label for="btc-ss" class="form-label">Screenshot (Optional)</label><input type="file" class="form-control" id="btc-ss" name="screenshot"></div>
                                    <button type="submit" class="btn btn-primary"><i class='bx bx-check me-1'></i> Submit Deposit</button>
                                </form>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($usdtWallets)): ?>
                            <div class="tab-pane fade <?php echo empty($btcWallets) ? 'show active' : ''; ?>" id="usdt-tab-pane" role="tabpanel" aria-labelledby="usdt-tab">
                                <div class="alert alert-info mb-4"><i class='bx bx-info-circle me-2'></i> <?php echo htmlspecialchars($usdtInstructions); ?></div>
                                <div class="mb-3">
                                    <label for="usdt-network" class="form-label">Select Network</label>
                                    <select class="form-select" id="usdt-network">
                                        <?php foreach (array_unique(array_column($usdtWallets, 'network')) as $network): ?>
                                            <option value="<?php echo htmlspecialchars($network); ?>"><?php echo htmlspecialchars($network); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <?php foreach ($usdtWallets as $wallet): ?>
                                    <div class="usdt-wallet-details" id="details-<?php echo htmlspecialchars($wallet['network']); ?>" style="display:none;">
                                        <div class="mb-3">
                                            <label class="form-label">USDT <?php echo htmlspecialchars($wallet['network']); ?> Address</label>
                                            <div class="input-group">
                                                <input type="text" class="form-control" value="<?php echo htmlspecialchars($wallet['wallet_address']); ?>" readonly>
                                                <button class="btn btn-outline-secondary copy-address" type="button" data-address="<?php echo htmlspecialchars($wallet['wallet_address']); ?>"><i class='bx bx-copy'></i> Copy</button>
                                            </div>
                                        </div>
                                        <div class="text-center mb-4"><div id="usdt-qrcode-<?php echo $wallet['id']; ?>" class="qrcode"></div></div>
                                    </div>
                                <?php endforeach; ?>
                                
                                <form method="post" enctype="multipart/form-data">
                                    <input type="hidden" name="currency" value="USDT">
                                    <input type="hidden" id="usdt-wallet-address" name="wallet_address">
                                    <input type="hidden" id="usdt-selected-network" name="network">
                                                                        <div class="mb-3"><label for="usdt-amount" class="form-label">Amount (USDT)</label><input type="number" class="form-control" id="usdt-amount" name="amount" step="0.01" min="1" required></div>
                                    <div class="mb-3"><label for="usdt-tx" class="form-label">Transaction Hash/ID</label><input type="text" class="form-control" id="usdt-tx" name="transaction_hash" required></div>
                                    <div class="mb-3"><label for="usdt-ss" class="form-label">Screenshot (Optional)</label><input type="file" class="form-control" id="usdt-ss" name="screenshot"></div>
                                    <button type="submit" class="btn btn-primary"><i class='bx bx-check me-1'></i> Submit Deposit</button>
                                </form>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Recent Deposits -->
        <div class="dashboard-card mt-4">
            <div class="card-header">
                <h5 class="card-title"><i class='bx bx-history'></i> Recent Deposits</h5>
            </div>
            <div class="card-body p-0">
                <?php if (empty($recent_deposits)): ?>
                    <div class="empty-state" style="padding: 3rem 1rem;">
                        <i class='bx bx-coin-stack empty-state-icon'></i>
                        <h5 class="empty-state-title">No Recent Deposits</h5>
                        <p class="empty-state-text">You haven't made any deposits yet.</p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Currency</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Transaction ID</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_deposits as $deposit): ?>
                                    <tr>
                                        <td><?php echo date('M d, Y H:i', strtotime($deposit['created_at'])); ?></td>
                                        <td>
                                            <?php if ($deposit['currency'] === 'BTC'): ?>
                                                <i class='bx bxl-bitcoin text-warning me-1'></i>
                                            <?php elseif ($deposit['currency'] === 'USDT'): ?>
                                                <i class='bx bx-dollar-circle text-success me-1'></i>
                                            <?php endif; ?>
                                            <?php echo htmlspecialchars($deposit['currency']); ?>
                                            <?php if ($deposit['network']): ?>
                                                <span class="badge bg-light text-dark"><?php echo htmlspecialchars($deposit['network']); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo number_format($deposit['amount'], 8); ?></td>
                                        <td>
                                            <?php if ($deposit['status'] === 'pending'): ?>
                                                <span class="badge bg-warning text-dark">Pending</span>
                                            <?php elseif ($deposit['status'] === 'approved'): ?>
                                                <span class="badge bg-success">Approved</span>
                                            <?php elseif ($deposit['status'] === 'rejected'): ?>
                                                <span class="badge bg-danger">Rejected</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="text-monospace text-truncate d-inline-block" style="max-width: 150px;"><?php echo htmlspecialchars($deposit['transaction_hash']); ?></span>
                                            <button class="btn btn-sm btn-link p-0 ms-2 copy-hash" data-hash="<?php echo htmlspecialchars($deposit['transaction_hash']); ?>" data-bs-toggle="tooltip" title="Copy Hash">
                                                <i class='bx bx-copy'></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Deposit Instructions -->
        <div class="dashboard-card mt-4">
            <div class="card-header">
                <h5 class="card-title"><i class='bx bx-info-circle'></i> How Deposits Work</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="d-flex">
                            <div class="me-3">
                                <div class="bg-primary-light text-primary rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                    <i class='bx bx-send'></i>
                                </div>
                            </div>
                            <div>
                                <h6 class="mb-1">Step 1: Send Cryptocurrency</h6>
                                <p class="text-muted mb-0">Send BTC or USDT to the provided wallet address from your external wallet or exchange.</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-4">
                        <div class="d-flex">
                            <div class="me-3">
                                <div class="bg-primary-light text-primary rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                    <i class='bx bx-receipt'></i>
                                </div>
                            </div>
                            <div>
                                <h6 class="mb-1">Step 2: Submit Transaction Details</h6>
                                <p class="text-muted mb-0">Enter the transaction hash/ID and amount you sent in the form above.</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-4">
                        <div class="d-flex">
                            <div class="me-3">
                                <div class="bg-primary-light text-primary rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                    <i class='bx bx-time'></i>
                                </div>
                            </div>
                            <div>
                                <h6 class="mb-1">Step 3: Wait for Confirmation</h6>
                                <p class="text-muted mb-0">Our team will verify your transaction on the blockchain (usually within 24 hours).</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-4">
                        <div class="d-flex">
                            <div class="me-3">
                                <div class="bg-primary-light text-primary rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                    <i class='bx bx-check-circle'></i>
                                </div>
                            </div>
                            <div>
                                <h6 class="mb-1">Step 4: Funds Credited</h6>
                                <p class="text-muted mb-0">Once verified, the funds will be credited to your account balance.</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="alert alert-warning mt-3 mb-0">
                    <i class='bx bx-error-circle me-2'></i>
                    <strong>Important:</strong> Always double-check the wallet address before sending. Cryptocurrency transactions are irreversible.
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/qrcodejs@1.0.0/qrcode.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Generate QR codes for BTC wallets
    <?php foreach ($btcWallets as $wallet): ?>
    if (document.getElementById("btc-qrcode-<?php echo $wallet['id']; ?>")) {
        new QRCode(document.getElementById("btc-qrcode-<?php echo $wallet['id']; ?>"), {
            text: "<?php echo $wallet['wallet_address']; ?>",
            width: 128,
            height: 128,
            colorDark: "#000000",
            colorLight: "#ffffff",
            correctLevel: QRCode.CorrectLevel.H
        });
    }
    <?php endforeach; ?>
    
    // Generate QR codes for USDT wallets
    <?php foreach ($usdtWallets as $wallet): ?>
    if (document.getElementById("usdt-qrcode-<?php echo $wallet['id']; ?>")) {
        new QRCode(document.getElementById("usdt-qrcode-<?php echo $wallet['id']; ?>"), {
            text: "<?php echo $wallet['wallet_address']; ?>",
            width: 128,
            height: 128,
            colorDark: "#000000",
            colorLight: "#ffffff",
            correctLevel: QRCode.CorrectLevel.H
        });
    }
    <?php endforeach; ?>
    
    // Copy wallet address to clipboard
    document.querySelectorAll('.copy-address').forEach(button => {
        button.addEventListener('click', function() {
            const address = this.dataset.address;
            navigator.clipboard.writeText(address).then(() => {
                // Change button text temporarily
                const originalHtml = this.innerHTML;
                this.innerHTML = '<i class="bx bx-check"></i> Copied!';
                
                // Reset button text after 2 seconds
                setTimeout(() => {
                    this.innerHTML = originalHtml;
                }, 2000);
            });
        });
    });
    
    // Copy transaction hash to clipboard
    document.querySelectorAll('.copy-hash').forEach(button => {
        button.addEventListener('click', function() {
            const hash = this.dataset.hash;
            navigator.clipboard.writeText(hash).then(() => {
                // Show tooltip
                const tooltip = bootstrap.Tooltip.getInstance(this);
                if (tooltip) {
                    tooltip.setContent({ '.tooltip-inner': 'Copied!' });
                    
                    // Reset tooltip after 2 seconds
                    setTimeout(() => {
                        tooltip.setContent({ '.tooltip-inner': 'Copy Hash' });
                    }, 2000);
                }
            });
        });
    });
    
    // Handle USDT network selection
    const usdtNetworkSelect = document.getElementById('usdt-network');
    if (usdtNetworkSelect) {
        // Function to update wallet details based on selected network
        function updateUsdtWalletDetails() {
            const selectedNetwork = usdtNetworkSelect.value;
            
            // Hide all wallet details
            document.querySelectorAll('.usdt-wallet-details').forEach(el => {
                el.style.display = 'none';
            });
            
            // Show selected network's wallet details
            const selectedDetails = document.getElementById('details-' + selectedNetwork);
            if (selectedDetails) {
                selectedDetails.style.display = 'block';
            }
            
            // Update form hidden fields
            <?php foreach ($usdtWallets as $wallet): ?>
            if (selectedNetwork === '<?php echo $wallet['network']; ?>') {
                document.getElementById('usdt-wallet-address').value = '<?php echo $wallet['wallet_address']; ?>';
                document.getElementById('usdt-selected-network').value = '<?php echo $wallet['network']; ?>';
            }
            <?php endforeach; ?>
        }
        
        // Set initial values
        updateUsdtWalletDetails();
        
        // Update when network selection changes
        usdtNetworkSelect.addEventListener('change', updateUsdtWalletDetails);
    }
    
    // Add hover effects to cards
    const cards = document.querySelectorAll('.dashboard-card');
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.boxShadow = '0 15px 30px -10px rgba(0,0,0,0.1), 0 4px 6px -4px rgba(0,0,0,0.1)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = '';
            this.style.boxShadow = '';
        });
    });
});
</script>

<style>
/* QR Code Container */
.qrcode {
    width: 128px;
    height: 128px;
    padding: 5px;
    background-color: white;
    border: 1px solid #e0e0e0;
    border-radius: 5px;
    margin: 0 auto;
}

/* Card Styles */
.dashboard-card {
    background: white;
    border-radius: 15px;
    border: none;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 1.5rem;
    transition: all 0.3s ease;
    overflow: hidden;
    position: relative;
}

.card-header {
    background: none;
    padding: 1.25rem;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.card-title {
    margin-bottom: 0;
    font-size: 1.1rem;
    font-weight: 600;
    display: flex;
    align-items: center;
}

.card-title i {
    margin-right: 0.5rem;
    font-size: 1.25rem;
    color: #4f46e5;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 2rem 1rem;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
}

.empty-state-icon {
    font-size: 2.5rem;
    color: #d1d5db;
    margin-bottom: 0.5rem;
}

.empty-state-title {
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 0.25rem;
}

.empty-state-text {
    color: #64748b;
    margin-bottom: 1rem;
    font-size: 0.875rem;
}

/* Badge Styles */
.badge {
    font-weight: 500;
    padding: 0.35em 0.65em;
    border-radius: 0.375rem;
}

/* Background Colors */
.bg-primary-light { background-color: rgba(79, 70, 229, 0.1); }
.text-primary { color: #4f46e5 !important; }
</style>

<?php include 'includes/footer.php'; ?>


