<?php
require_once 'includes/auth_check.php';
$page = 'settings';
require_once '../config/database.php';
require_once '../includes/functions.php';

$database = new Database();
$db = $database->getConnection();

// --- First, let's check the actual structure of your settings table ---
try {
    // This query will tell us the column names in your settings table
    $columnsQuery = $db->query("SHOW COLUMNS FROM settings");
    $columns = $columnsQuery->fetchAll(PDO::FETCH_COLUMN);
    
    // Determine the correct column names based on what exists in the database
    $keyColumn = in_array('setting_name', $columns) ? 'setting_name' : 
                (in_array('setting_key', $columns) ? 'setting_key' : '');
    
    $valueColumn = in_array('setting_value', $columns) ? 'setting_value' : 
                  (in_array('value', $columns) ? 'value' : '');
    
    // If we couldn't find the right columns, we'll create a new table
    if (empty($keyColumn) || empty($valueColumn)) {
        throw new Exception("Could not determine settings table structure");
    }
} catch (Exception $e) {
    // If there's any error (including table not existing), create the table
    try {
        $db->exec("DROP TABLE IF EXISTS settings");
        $db->exec("CREATE TABLE settings (
            id INT AUTO_INCREMENT PRIMARY KEY,
            setting_key VARCHAR(100) NOT NULL,
            setting_value TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_setting_key (setting_key)
        )");
        
        // Set the column names for our new table
        $keyColumn = 'setting_key';
        $valueColumn = 'setting_value';
        
        // Insert default settings
        $defaultSettings = [
            ['site_name', 'BlueSeal Bank System'],
            ['site_contact_email', 'support@example.com'],
            ['theme_primary_color', '#4f46e5'],
            ['theme_sidebar_bg', '#111827'],
            ['theme_sidebar_text', '#ffffff'],
            ['session_timeout', '30'],
            ['max_login_attempts', '5']
        ];
        
        $insertStmt = $db->prepare("INSERT INTO settings ($keyColumn, $valueColumn) VALUES (?, ?)");
        foreach ($defaultSettings as $setting) {
            $insertStmt->execute($setting);
        }
    } catch (PDOException $createError) {
        die("Could not create settings table: " . $createError->getMessage());
    }
}

// --- Handle Form Submission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $_SESSION['error_message'] = "Invalid security token. Please try again.";
    } else {
        try {
            // Begin transaction
            $db->beginTransaction();
            
            // Use the correct column names in our query
            $stmt = $db->prepare("INSERT INTO settings ($keyColumn, $valueColumn) VALUES (?, ?) 
                                 ON DUPLICATE KEY UPDATE $valueColumn = ?");
            
            foreach ($_POST as $key => $value) {
                if ($key === 'csrf_token' || $key === 'submit') continue;
                $sanitized_value = trim($value);
                $stmt->execute([$key, $sanitized_value, $sanitized_value]);
            }
            
            $db->commit();
            $_SESSION['success_message'] = "Settings have been updated successfully.";
            
        } catch (PDOException $e) {
            $db->rollBack();
            error_log("Settings Update Error: " . $e->getMessage());
            $_SESSION['error_message'] = "Database error: " . $e->getMessage();
        }
    }
    
    header("Location: settings.php");
    exit;
}

// --- Fetch all settings using the correct column names ---
$settings = [];
try {
    $stmt = $db->query("SELECT $keyColumn, $valueColumn FROM settings");
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row[$keyColumn]] = $row[$valueColumn];
    }
} catch (PDOException $e) {
    error_log("Settings Fetch Error: " . $e->getMessage());
}

// Generate a CSRF token for the form
$_SESSION['csrf_token'] = bin2hex(random_bytes(32));

include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <h1 class="h3 mb-4 text-gray-800">System Settings</h1>

    <!-- Display Success/Error Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <div class="card shadow mb-4">
        <div class="card-body">
            <form action="settings.php" method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">

                <!-- Tab Navigation -->
                <ul class="nav nav-tabs" id="settingsTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="general-tab" data-bs-toggle="tab" data-bs-target="#general-pane" type="button" role="tab">General</button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="theme-tab" data-bs-toggle="tab" data-bs-target="#theme-pane" type="button" role="tab">Theme</button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="security-tab" data-bs-toggle="tab" data-bs-target="#security-pane" type="button" role="tab">Security</button>
                    </li>
                </ul>

                <!-- Tab Content -->
                <div class="tab-content pt-4" id="settingsTabsContent">
                    <!-- General Settings Tab -->
                    <div class="tab-pane fade show active" id="general-pane" role="tabpanel" aria-labelledby="general-tab">
                        <h5 class="mb-3">General Site Settings</h5>
                        <div class="mb-3">
                            <label for="site_name" class="form-label">Site Name</label>
                            <input type="text" class="form-control" id="site_name" name="site_name" value="<?php echo htmlspecialchars($settings['site_name'] ?? 'BlueSeal Bank System'); ?>">
                        </div>
                        <div class="mb-3">
                            <label for="site_contact_email" class="form-label">Public Contact Email</label>
                            <input type="email" class="form-control" id="site_contact_email" name="site_contact_email" value="<?php echo htmlspecialchars($settings['site_contact_email'] ?? 'support@example.com'); ?>">
                        </div>
                    </div>

                    <!-- Theme Settings Tab -->
                    <div class="tab-pane fade" id="theme-pane" role="tabpanel" aria-labelledby="theme-tab">
                        <h5 class="mb-3">Theme & Color Settings</h5>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label for="theme_primary_color" class="form-label">Primary Color</label>
                                <input type="color" class="form-control form-control-color w-100" id="theme_primary_color" name="theme_primary_color" value="<?php echo htmlspecialchars($settings['theme_primary_color'] ?? '#4f46e5'); ?>">
                                <small class="text-muted">Used for buttons, links, and highlights</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="theme_sidebar_bg" class="form-label">Sidebar Background</label>
                                <input type="color" class="form-control form-control-color w-100" id="theme_sidebar_bg" name="theme_sidebar_bg" value="<?php echo htmlspecialchars($settings['theme_sidebar_bg'] ?? '#111827'); ?>">
                                <small class="text-muted">Background color of the sidebar</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="theme_sidebar_text" class="form-label">Sidebar Text</label>
                                <input type="color" class="form-control form-control-color w-100" id="theme_sidebar_text" name="theme_sidebar_text" value="<?php echo htmlspecialchars($settings['theme_sidebar_text'] ?? '#ffffff'); ?>">
                                <small class="text-muted">Text color in the sidebar</small>
                            </div>
                        </div>
                    </div>

                    <!-- Security Settings Tab -->
                    <div class="tab-pane fade" id="security-pane" role="tabpanel" aria-labelledby="security-tab">
                        <h5 class="mb-3">Security Settings</h5>
                        <div class="mb-3">
                            <label for="session_timeout" class="form-label">Session Timeout (minutes)</label>
                            <input type="number" class="form-control" id="session_timeout" name="session_timeout" value="<?php echo htmlspecialchars($settings['session_timeout'] ?? '30'); ?>">
                        </div>
                        <div class="mb-3">
                            <label for="max_login_attempts" class="form-label">Max Login Attempts</label>
                            <input type="number" class="form-control" id="max_login_attempts" name="max_login_attempts" value="<?php echo htmlspecialchars($settings['max_login_attempts'] ?? '5'); ?>">
                        </div>
                    </div>
                </div>

                <div class="mt-4 border-top pt-4">
                    <button type="submit" name="submit" class="btn btn-primary">Save All Settings</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
